
#import "noticeDB.h"
#import <sqlite3.h>

static sqlite3 *database = NULL;
static sqlite3_stmt *insert_statement = NULL;
static sqlite3_stmt *update_statement = NULL;
static sqlite3_stmt *delete_statement = NULL;
static sqlite3_stmt *select_statement = NULL;

sqlite3 *openNoticeDB(void) {
    NSCAssert2([NSThread isMainThread], @"%s at line %d called on secondary thread", __FUNCTION__, __LINE__);
    if (database == NULL) {
        BOOL success;
        NSFileManager *fileManager = [NSFileManager defaultManager];
        NSError *error;
        NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        NSString *documentsDirectory = [paths objectAtIndex:0];
        NSString *writableDBPath = [documentsDirectory stringByAppendingPathComponent:@"notice.sqlite"];
        if ([fileManager fileExistsAtPath:writableDBPath] == NO) {
            NSString *defaultDBPath = [[NSBundle mainBundle] pathForResource:@"notice" ofType:@"sqlite"];
            success = [fileManager copyItemAtPath:defaultDBPath toPath:writableDBPath error:&error];
            if (!success) {
                NSCAssert1(0, @"Failed to create writable database file with message '%@'.", [error localizedDescription]);
            }
        }
        if (sqlite3_open([writableDBPath UTF8String], &database) != SQLITE_OK) {
            sqlite3_close(database);
            database = NULL;
            NSCAssert1(0, @"Failed to open database with message '%s'.", sqlite3_errmsg(database));
        }
    }
    return database;
}

void closeNoticeDB() {
    NSCAssert2([NSThread isMainThread], @"%s at line %d called on secondary thread", __FUNCTION__, __LINE__);
    if (insert_statement != NULL) {
        sqlite3_finalize(insert_statement);
        insert_statement = NULL;
    }
    if (update_statement != NULL) {
        sqlite3_finalize(update_statement);
        update_statement = NULL;
    }
    if (delete_statement != NULL) {
        sqlite3_finalize(delete_statement);
        delete_statement = NULL;
    }
    if (select_statement != NULL) {
        sqlite3_finalize(select_statement);
        select_statement = NULL;
    }
    if (database == NULL) return;
    if (sqlite3_close(database) != SQLITE_OK) {
        NSCAssert1(0, @"Error: failed to close database with message '%s'.", sqlite3_errmsg(database));
    }
    database = NULL;
}

void insertNoticeDB(notice *noticeTmp) {
    sqlite3 *db = openNoticeDB();
    insert_statement = NULL;
	static const char *sql = "insert into notice_demo (rowid, subject, reg_date, content) values (?, ?, ?, ?)";
	sqlite3_prepare_v2(db, sql, -1, &insert_statement, NULL);
    sqlite3_bind_int(insert_statement, 1, [noticeTmp.rowid intValue]);   
	sqlite3_bind_text(insert_statement, 2, [noticeTmp.subject UTF8String], -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(insert_statement, 3, [noticeTmp.reg_date UTF8String], -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(insert_statement, 4, [noticeTmp.content UTF8String], -1, SQLITE_TRANSIENT);
    int success = sqlite3_step(insert_statement);
    sqlite3_reset(insert_statement);
    if (success == SQLITE_ERROR) {
        NSCAssert1(0, @"Error: failed to insert into the database with message '%s'.", sqlite3_errmsg(db));
    }
	closeNoticeDB();
}

void updateNoticeDB(notice *noticeTmp) {
    sqlite3 *db = openNoticeDB();
    update_statement = NULL;
    static const char *sql = "update notice_demo set subject=?, reg_date=?, content=? where rowid=?";
    sqlite3_prepare_v2(db, sql, -1, &update_statement, NULL);
    sqlite3_bind_text(update_statement, 1, [noticeTmp.subject UTF8String], -1, NULL);
    sqlite3_bind_text(update_statement, 2, [noticeTmp.reg_date UTF8String], -1, NULL);
    sqlite3_bind_text(update_statement, 3, [noticeTmp.content UTF8String], -1, NULL);
	sqlite3_bind_int(update_statement, 4, [noticeTmp.rowid intValue]);
    int success = sqlite3_step(update_statement);
    sqlite3_reset(update_statement);
    if (success == SQLITE_ERROR) {
        NSCAssert1(0, @"Error: failed to insert into the database with message '%s'.", sqlite3_errmsg(db));
    }
	closeNoticeDB();
}

void replaceNoticeDB(notice *noticeTmp) {
    sqlite3 *db = openNoticeDB();

	int existId = 0;
    static const char *sql = "select count(*) from notice_demo where rowid = ? ";
    sqlite3_prepare_v2(db, sql, -1, &select_statement, NULL);
	sqlite3_bind_int(select_statement, 1, [noticeTmp.rowid intValue]);
	if (sqlite3_step(select_statement)==SQLITE_ROW) {
		existId = sqlite3_column_int(select_statement, 0);
	}
	sqlite3_reset(select_statement);
	closeNoticeDB();
	
	if (existId>0) {
        updateNoticeDB(noticeTmp);
    } else {
		insertNoticeDB(noticeTmp);
	}
	
}

void deleteNoticeDB(void) {
    sqlite3 *db = openNoticeDB();
    if (delete_statement == NULL) {
        static const char *sql = "delete from notice_demo";
        if (sqlite3_prepare_v2(db, sql, -1, &delete_statement, NULL) != SQLITE_OK) {
            NSCAssert1(0, @"Error: failed to prepare statement with message '%s'.", sqlite3_errmsg(db));
        }
    }
    int success = sqlite3_step(delete_statement);
    if (success == SQLITE_ERROR) {
        NSCAssert1(0, @"Error: failed to execute query with message '%s'.", sqlite3_errmsg(db));
    }
    sqlite3_reset(delete_statement);
	closeNoticeDB();
}

void deleteRowNoticeDB(int rowid) {
    sqlite3 *db = openNoticeDB();
    delete_statement = NULL;
    static const char *sql = "delete from notice_demo where rowid = ? ";
    sqlite3_prepare_v2(db, sql, -1, &delete_statement, NULL);
	sqlite3_bind_int(delete_statement, 1, rowid);
    sqlite3_step(delete_statement);
    sqlite3_reset(delete_statement);
	closeNoticeDB();
}

NSMutableArray *selectNoticeDB(void) {
    sqlite3 *db = openNoticeDB();
    select_statement = NULL;
	static const char *sql = "select * from notice_demo order by rowid desc";
    sqlite3_prepare_v2(db, sql, -1, &select_statement, NULL);
	NSMutableArray *noticeData = [NSMutableArray array];
    while (sqlite3_step(select_statement)==SQLITE_ROW) {
		notice *noticeTmp = [[notice alloc] init];
		
		NSString *rowid = [[NSString alloc] initWithFormat:@"%d", sqlite3_column_int(select_statement, 0)];
		NSString *subject = [[NSString alloc] initWithUTF8String: (char *) sqlite3_column_text(select_statement, 1)];
		NSString *reg_date = [[NSString alloc] initWithUTF8String: (char *) sqlite3_column_text(select_statement, 2)];
		NSString *content = [[NSString alloc] initWithUTF8String: (char *) sqlite3_column_text(select_statement, 3)];
		
		noticeTmp.rowid = rowid;
		noticeTmp.subject = subject;
		noticeTmp.reg_date = reg_date;
		noticeTmp.content = content;
		
		[noticeData addObject:noticeTmp];
		
		[rowid release];[subject release];[reg_date release];[content release];
		[noticeTmp release];
		
	}
    sqlite3_reset(select_statement);
	closeNoticeDB();
    return noticeData;
}

